<?php
/**
 * Plugin Name: Location SEO
 * Plugin URI: https://capitaldistrictdigital.com/
 * Description: Harness the power of AI with Location SEO! 
 * Version: 2.1
 * Author: Capital District Digital
 * Author URI: https://capitaldistrictdigital.com/
 * Text Domain: location-seo
 */

class Location_SEO {
    private $google_maps_api_key;
    private $openai_api_key;
    private $openai_organization_id;
    private $company_name;
    private $telephone;

    public function __construct() {
        $this->google_maps_api_key = get_option('location_seo_google_maps_api_key');
        $this->openai_api_key = get_option('location_seo_openai_api_key');
        $this->openai_organization_id = get_option('location_seo_openai_organization_id');
        $this->company_name = get_option('location_seo_company_name');
        $this->telephone = get_option('location_seo_telephone');

        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'handle_form_submission'));
        add_shortcode('location_seo_map', array($this, 'generate_map_shortcode'));
    }

    public function add_admin_menu() {
        add_menu_page('Location SEO', 'Location SEO', 'manage_options', 'location-seo', array($this, 'render_admin_page'), 'dashicons-location', 6);
    }

    public function render_admin_page() {
        ?>
        <h2>Location SEO</h2>
        <form method="POST">
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="company_name">Company Name</label></th>
                    <td><input type="text" name="company_name" id="company_name" value="<?= get_option('location_seo_company_name'); ?>" placeholder="Enter your company name"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="telephone">Telephone</label></th>
                    <td><input type="text" name="telephone" id="telephone" value="<?= get_option('location_seo_telephone'); ?>" placeholder="Enter your telephone number"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="keyphrase">Keyphrase</label></th>
                    <td><input type="text" name="keyphrase" id="keyphrase" placeholder="Enter your keyphrase"></td>
                </tr>
                <?php for ($i = 1; $i <= 3; $i++) : ?>
                    <tr>
                        <th scope="row"><label for="location<?= $i ?>">Location <?= $i ?></label></th>
                        <td><input type="text" name="locations[]" id="location<?= $i ?>" placeholder="Enter a location"></td>
                    </tr>
                <?php endfor; ?>
                <tr>
                    <th scope="row"><label for="google_maps_api_key">Google Maps API Key</label></th>
                    <td><input type="text" name="google_maps_api_key" id="google_maps_api_key" value="<?= get_option('location_seo_google_maps_api_key'); ?>" placeholder="Enter your Google Maps API Key"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="openai_api_key">OpenAI API Key</label></th>
                    <td><input type="text" name="openai_api_key" id="openai_api_key" value="<?= get_option('location_seo_openai_api_key'); ?>" placeholder="Enter your OpenAI API Key"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="openai_organization_id">OpenAI Organization ID</label></th>
                    <td><input type="text" name="openai_organization_id" id="openai_organization_id" value="<?= get_option('location_seo_openai_organization_id'); ?>" placeholder="Enter your OpenAI Organization ID"></td>
                </tr>
            </table>
            <p class="submit">
            <input type="submit" name="location_seo_form_submit" id="submit_pages" class="button button-primary" value="Generate Pages">
            <input type="submit" name="location_seo_form_submit" id="submit_posts" class="button button-primary" value="Generate Blog Posts">
        </p>
            <p class="submit"><input type="submit" name="location_seo_form_submit" id="submit" class="button button-primary" value="Save Changes"></p>
        </form>
        <?php
    }

public function handle_form_submission() {
    if (isset($_POST['location_seo_form_submit'])) {
        // Your existing code for handling form data
        $company_name = sanitize_text_field($_POST['company_name']);
        $keyphrase = sanitize_text_field($_POST['keyphrase']);
        $locations = array_filter(array_map('sanitize_text_field', $_POST['locations']));
        $this->google_maps_api_key = sanitize_text_field($_POST['google_maps_api_key']);
        $this->openai_api_key = sanitize_text_field($_POST['openai_api_key']);
        $this->openai_organization_id = sanitize_text_field($_POST['openai_organization_id']);
        $this->company_name = sanitize_text_field($_POST['company_name']);
        $this->telephone = sanitize_text_field($_POST['telephone']);

        update_option('location_seo_google_maps_api_key', $this->google_maps_api_key);
        update_option('location_seo_openai_api_key', $this->openai_api_key);
        update_option('location_seo_openai_organization_id', $this->openai_organization_id);
        update_option('location_seo_company_name', $this->company_name);
        update_option('location_seo_telephone', $this->telephone);

        if ($_POST['location_seo_form_submit'] == 'Generate Pages') {
            foreach ($locations as $location) {
                // Logic to generate pages
                $content = $this->generate_content($keyphrase, $location, $company_name, 'Generate Pages');
                $location_schema = $this->generate_location_schema($location);

                $page_id = wp_insert_post(array(
                    'post_title' => $keyphrase . ' ' . $location,
                    'post_content' => $content . '[location_seo_map address="' . $location . '"]' . $location_schema,
                    'post_status' => 'draft',
                    'post_type' => 'page'
                ));

                // Handle errors or post creation confirmation here
            }
        } elseif ($_POST['location_seo_form_submit'] == 'Generate Blog Posts') {
            foreach ($locations as $location) {
                // Logic to generate blog posts
                $content = $this->generate_content($keyphrase, $location, $company_name, 'Generate Blog Posts');
                $location_schema = $this->generate_location_schema($location);

                $post_id = wp_insert_post(array(
                    'post_title' => $keyphrase . ' ' . $location,
                    'post_content' => $content . '[location_seo_map address="' . $location . '"]' . $location_schema,
                    'post_status' => 'draft',
                    'post_type' => 'post'
                ));

                // Handle errors or post creation confirmation here
            }
        }
    }
}

    public function generate_map_shortcode($atts) {
        $atts = shortcode_atts(array('address' => ''), $atts, 'location_seo_map');
        $address = urlencode($atts['address']);

        return '<iframe width="100%" height="450" style="border:0" loading="lazy" src="https://www.google.com/maps/embed/v1/place?q=' . $address . '&key=' . $this->google_maps_api_key . '"></iframe>';
    }

    private function generate_content($keyphrase, $location, $company_name, $content_type) {
        // Define the API endpoint
        $api_url = 'https://api.openai.com/v1/chat/completions';

        // Define the headers
        $headers = array(
            'Authorization' => 'Bearer ' . $this->openai_api_key,
            'OpenAI-Organization' => $this->openai_organization_id,
            'Content-Type' => 'application/json'
        );

        // Define the assistant message based on the content type
        if ($content_type === 'Generate Pages') {
        $assistant_message = 'Write a 500-word well-written, engaging, and useful content SEO optimized page rendered in HTML starting with the <h1> tag. Distribute the keyphrase, semantic keywords related to this keyphrase and location throughout the content, headings and subheadings evenly. Include headings, subheadings, bullet points, lists, our Services, process and benefits. Location history, Interesting facts and Local culture and traditions about [Location]. Generate a list of frequently asked questions and their answers about the services. Each question should start with "Q:" and be wrapped in an <h3> tag, and each answer should start with "A:" and be displayed right below its corresponding question. This FAQ section should be specific to [Location] and about "' . $keyphrase . '" in "' . $location . '" for ' . $company_name . '.';
        } elseif ($content_type === 'Generate Blog Posts') {
            $assistant_message = 'Write a 500 to 600 word well-written, engaging, and useful SEO optimize blog post rendered in HTML. Distribute the keyphrase, semantic keywords related to this keyphrase and location throughout the content, headings and subheadings evenly. Include headings, subheadings, detailed main content, bullet points and lists. Generate a list of frequently asked questions and their answers about the services. Each question should start with "Q:" and be wrapped in an <h3> tag, and each answer should start with "A:" and be displayed right below its corresponding question about "' . $keyphrase . '" in "' . $location . '" for ' . $company_name . '.';
        } else {
            return 'Invalid content type';
        }

        // Define the request body
        $body = json_encode(array(
            'model' => 'gpt-3.5-turbo',
            'messages' => array(
                array('role' => 'user', 'content' => $assistant_message)
            ),
            'temperature' => 0.7
        ));

        // Make the API request
        $response = wp_remote_post($api_url, array(
            'headers' => $headers,
            'body' => $body,
            'timeout' => 80 // Increased timeout to 80 seconds
        ));

        // Check for errors
        if (is_wp_error($response)) {
            return 'Error: ' . $response->get_error_message();
        }

        // Parse the response
        $response_body = json_decode(wp_remote_retrieve_body($response), true);

        // Extract the assistant's response from the choices array
        $assistant_response = $response_body['choices'][0]['message']['content'];

        // Remove unwanted code and characters
        $assistant_response = preg_replace('/^\s*\<\!DOCTYPE.*?\<h1\>/is', '<h1>', $assistant_response);
        $assistant_response = preg_replace('/\<meta[^>]+\>/is', '', $assistant_response);
        $assistant_response = trim($assistant_response, "\n\r");

        // Return the assistant's response
        return $assistant_response;
    }

    private function generate_location_schema($location) {
        $location_data = $this->get_location_data($location);
        $state = $location_data['state'];
        $city = $location_data['city'];
        $country = $location_data['country'];

        $location_schema = '<script type="application/ld+json">' .
            '{' .
            '"@context": "http://schema.org",' .
            '"@type": "LocalBusiness",' .
            '"name": "' . $this->company_name . '",' .
            '"telephone": "' . $this->telephone . '",' .
            '"address": {' .
            '"@type": "PostalAddress",' .
            '"addressLocality": "' . $city . '",' .
            '"addressRegion": "' . $state . '",' .
            '"addressCountry": "' . $country . '"' .
            '}' .
            '}' .
            '</script>';

        return $location_schema;
    }

    private function get_location_data($location) {
        $api_url = 'https://maps.googleapis.com/maps/api/geocode/json?address=' . urlencode($location) . '&key=' . $this->google_maps_api_key;

        $response = wp_remote_get($api_url);

        if (is_wp_error($response)) {
            return array(
                'state' => '',
                'city' => '',
                'country' => ''
            );
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if ($data['status'] === 'OK') {
            $result = $data['results'][0];
            $address_components = $result['address_components'];

            $state = '';
            $city = '';
            $country = '';

            foreach ($address_components as $component) {
                $types = $component['types'];
                if (in_array('administrative_area_level_1', $types)) {
                    $state = $component['short_name'];
                } elseif (in_array('locality', $types)) {
                    $city = $component['long_name'];
                } elseif (in_array('country', $types)) {
                    $country = $component['long_name'];
                }
            }

            return array(
                'state' => $state,
                'city' => $city,
                'country' => $country
            );
        }

        return array(
            'state' => '',
            'city' => '',
            'country' => ''
        );
    }
}

$location_seo = new Location_SEO();

register_activation_hook(__FILE__, function() {
    add_option('location_seo_google_maps_api_key', '');
    add_option('location_seo_openai_api_key', '');
    add_option('location_seo_openai_organization_id', '');
    add_option('location_seo_company_name', '');
    add_option('location_seo_telephone', '');
});