<?php
/**
 * Plugin Name: SEO Content Generator
 * Plugin URI: https://capitaldistrictdigital.com/
 * Description: Generate SEO-optimized content for blogs and pages. Now includes a meta box for posts and pages.
 * Version: 2.6  
 * Author: Capital District Digital
 * Author URI: https://capitaldistrictdigital.com/
 * Text Domain: seo-content-generator
 *
 * @version 2.6  // Marking this version for future reference
 * Added Features:
 * - Meta box now available in both Post and Page editors.
 * - Enhanced form submission handling for saving settings and generating content.
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

class SEO_Content_Generator {
    private $openai_api_key;
    private $openai_organization_id;
    private $company_name;

    public function __construct() {
        $this->openai_api_key = get_option('seo_content_generator_openai_api_key');
        $this->openai_organization_id = get_option('seo_content_generator_openai_organization_id');
        $this->company_name = get_option('seo_content_generator_company_name');

        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'handle_form_submission'));
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
    }

    public function run() {
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    public function activate() {
        add_option('seo_content_generator_openai_api_key', '');
        add_option('seo_content_generator_openai_organization_id', '');
        add_option('seo_content_generator_company_name', '');
    }

    public function deactivate() {
        delete_option('seo_content_generator_openai_api_key');
        delete_option('seo_content_generator_openai_organization_id');
        delete_option('seo_content_generator_company_name');
    }

    public function add_admin_menu() {
        add_menu_page('SEO Content Generator', 'SEO Content Generator', 'manage_options', 'seo-content-generator', array($this, 'render_admin_page'), 'dashicons-admin-site');
    }

    public function render_admin_page() {
        ?>
        <h2>SEO Content Generator</h2>
        <form method="POST">
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="company_name">Company Name</label></th>
                    <td><input type="text" name="company_name" id="company_name" value="<?= esc_attr(get_option('seo_content_generator_company_name')); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="keyphrase">Keyphrase</label></th>
                    <td><input type="text" name="keyphrase" id="keyphrase" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="openai_api_key">OpenAI API Key</label></th>
                    <td><input type="text" name="openai_api_key" id="openai_api_key" value="<?= esc_attr(get_option('seo_content_generator_openai_api_key')); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="openai_organization_id">OpenAI Organization ID</label></th>
                    <td><input type="text" name="openai_organization_id" id="openai_organization_id" value="<?= esc_attr(get_option('seo_content_generator_openai_organization_id')); ?>" class="regular-text" /></td>
                </tr>
                <?php wp_nonce_field('seo_content_generator_nonce', 'seo_content_generator_nonce_field'); ?>
            </table>
            <p class="submit">
                <input type="submit" name="seo_settings_submit" id="save_settings" class="button button-primary" value="Save Settings">
                <input type="submit" name="seo_content_form_submit" id="submit_pages" class="button button-primary" value="Generate Pages">
                <input type="submit" name="seo_content_form_submit" id="submit_posts" class="button button-primary" value="Generate Blog Posts">
            </p>
        </form>
        <?php
    }

public function add_meta_box() {
    // Add meta box for pages
    add_meta_box(
        'seo_content_generator_meta_box',
        'SEO Content Generator',
        array($this, 'render_meta_box_content'),
        'page', // Post type: page
        'normal',
        'high'
    );

    // Add the same meta box for posts
    add_meta_box(
        'seo_content_generator_meta_box',
        'SEO Content Generator',
        array($this, 'render_meta_box_content'),
        'post', // Post type: post
        'normal',
        'high'
    );
}

    public function render_meta_box_content($post) {
        wp_nonce_field('seo_content_generator_nonce', 'seo_content_generator_nonce_field');
        ?>
        <p>
            <label for="meta_box_company_name">Company Name:</label>
            <input type="text" name="company_name" id="meta_box_company_name" value="<?= esc_attr(get_option('seo_content_generator_company_name')); ?>" class="regular-text" />
        </p>
        <p>
            <label for="meta_box_keyphrase">Keyphrase:</label>
            <input type="text" name="keyphrase" id="meta_box_keyphrase" class="regular-text" />
        </p>
        <p>
            <input type="submit" name="seo_content_form_submit" class="button button-primary" value="Generate Pages">
            <input type="submit" name="seo_content_form_submit" class="button button-primary" value="Generate Blog Posts">
        </p>
        <?php
    }

    public function handle_form_submission() {
        if (isset($_POST['seo_settings_submit']) && check_admin_referer('seo_content_generator_nonce', 'seo_content_generator_nonce_field')) {
            // Save settings
            update_option('seo_content_generator_openai_api_key', sanitize_text_field($_POST['openai_api_key']));
            update_option('seo_content_generator_openai_organization_id', sanitize_text_field($_POST['openai_organization_id']));
            update_option('seo_content_generator_company_name', sanitize_text_field($_POST['company_name']));
            // Optionally add admin notice for settings saved
        } elseif (isset($_POST['seo_content_form_submit']) && check_admin_referer('seo_content_generator_nonce', 'seo_content_generator_nonce_field')) {
            // Generate content
            $company_name = sanitize_text_field($_POST['company_name']);
            $keyphrase = sanitize_text_field($_POST['keyphrase']);
            $content = $this->generate_content($keyphrase, $company_name, $_POST['seo_content_form_submit']);

            if ($_POST['seo_content_form_submit'] === 'Generate Pages') {
                wp_insert_post(array(
                    'post_title' => $keyphrase,
                    'post_content' => $content,
                    'post_status' => 'draft',
                    'post_type' => 'page'
                ));
            } elseif ($_POST['seo_content_form_submit'] === 'Generate Blog Posts') {
                wp_insert_post(array(
                    'post_title' => $keyphrase,
                    'post_content' => $content,
                    'post_status' => 'draft',
                    'post_type' => 'post'
                ));
            }
        }
    }

    private function generate_content($keyphrase, $company_name, $content_type) {
        $api_url = 'https://api.openai.com/v1/chat/completions';

        $headers = array(
            'Authorization' => 'Bearer ' . $this->openai_api_key,
            'Content-Type' => 'application/json'
        );

        $assistant_message = 'Write a 600-word well-written, engaging, and useful content SEO optimized rendered in HTML starting with the <h1> tag. Distribute the keyphrase, semantic keywords related to this keyphrase throughout the content, headings and subheadings evenly. Include headings, subheadings, bullet points and lists about "' . $keyphrase . '" for ' . $company_name . '.';

        $body = json_encode(array(
            'model' => 'gpt-3.5-turbo',
            'messages' => array(
                array('role' => 'user', 'content' => $assistant_message)
            ),
            'temperature' => 0.7
        ));

        $response = wp_remote_post($api_url, array(
            'headers' => $headers,
            'body' => $body,
            'timeout' => 80 // Increased timeout to 80 seconds
        ));

        if (is_wp_error($response)) {
            return 'Error: ' . $response->get_error_message();
        }

        $response_body = json_decode(wp_remote_retrieve_body($response), true);
        return $response_body['choices'][0]['message']['content'];
    }
}

$seo_content_generator = new SEO_Content_Generator();
$seo_content_generator->run();
