<?php
/**
 * Plugin Name: OpenAI Yoast SEO
 * Description: A plugin to generate meta title and description using OpenAI for Yoast SEO.
 * Version: 2.0
 * Author: Capital District Digital
 */

// Ensure the file is not accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

class OpenAI_Yoast_SEO {
    private $openai_api_key;
    private $openai_organization_id;

    public function __construct() {
        $this->openai_api_key = get_option('openai_api_key');
        $this->openai_organization_id = get_option('openai_organization_id');

        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_generate_meta_data', array($this, 'ajax_generate_meta_data'));
        add_action('edit_form_after_title', array($this, 'add_button'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }

    public function enqueue_scripts() {
        wp_enqueue_script('openai-yoast-seo', plugins_url('js/script.js', __FILE__), array('jquery'), null, true );
        wp_localize_script('openai-yoast-seo', 'openai_yoast_seo', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('openai-yoast-seo-nonce')
        ));
    }

    public function add_button() {
        echo '<button id="generate-meta-data" class="button">Generate Meta Data</button>';
    }

    public function generate_meta_data($post_id) {
        $post = get_post($post_id);
        $content = $post->post_content;

        $api_url = 'https://api.openai.com/v1/chat/completions';

        $headers = array(
            'Authorization' => 'Bearer ' . $this->openai_api_key,
            'OpenAI-Organization' => $this->openai_organization_id,
            'Content-Type' => 'application/json'
        );

        $body = json_encode(array(
            'model' => 'gpt-3.5-turbo',
            'messages' => array(
                array('role' => 'user', 'content' => "Generate a meta title and a meta description for the following content: {$content}")
            ),
            'temperature' => 0.7
        ));

        $response = wp_remote_post($api_url, array(
            'headers' => $headers,
            'body' => $body,
            'timeout' => 80
        ));

        if (is_wp_error($response)) {
            error_log('OpenAI Error: ' . $response->get_error_message());
            return array('error' => $response->get_error_message());
        }

        $response_body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($response_body['choices'][0]['message']['content'])) {
            $assistant_response = $response_body['choices'][0]['message']['content'];

            list($meta_title, $meta_description) = explode("\n", $assistant_response, 2);

            $meta_title = str_replace('Meta Title:', '', $meta_title);
            $meta_description = str_replace('Meta Description:', '', $meta_description);

            $meta_description = $this->trim_description(trim($meta_description), 147);

            update_post_meta($post_id, '_yoast_wpseo_title', trim($meta_title));
            update_post_meta($post_id, '_yoast_wpseo_metadesc', trim($meta_description));

            return array('success' => true);
        } else {
            error_log('Unexpected API response: ' . print_r($response_body, true));
            return array('error' => 'Unexpected API response');
        }
    }

    private function trim_description($description, $char_limit) {
        $min_length = 132;

        while (strlen($description) < $min_length) {
            $description .= ' ' . $description;
        }

        if (strlen($description) > $char_limit) {
            $description = substr($description, 0, $char_limit);
            $last_space_pos = strrpos($description, ' ');

            if ($last_space_pos !== false) {
                $description = substr($description, 0, $last_space_pos);
            }
        }

        return $description;
    }

    public function ajax_generate_meta_data() {
        check_ajax_referer('openai-yoast-seo-nonce', 'nonce');

        $post_id = intval($_POST['post_id']);
        if ($post_id > 0) {
            $result = $this->generate_meta_data($post_id);
            if (isset($result['success'])) {
                wp_send_json_success(array('message' => 'Meta data generated successfully.'));
            } else {
                wp_send_json_error(array('message' => 'Error: ' . $result['error']));
            }
        } else {
            error_log('Invalid post ID received: ' . print_r($_POST, true));
            wp_send_json_error(array('message' => 'Invalid post ID.'));
        }
    }

    public function add_admin_menu() {
        add_options_page(
            'OpenAI Yoast SEO Settings',
            'OpenAI Yoast SEO',
            'manage_options',
            'openai_yoast_seo_settings',
            array($this, 'settings_page')
        );
    }

    public function settings_page() {
        ?>
        <div class="wrap">
        <h1>OpenAI Yoast SEO Settings</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('openai_yoast_seo');
            do_settings_sections('openai_yoast_seo');
            ?>
            <table class="form-table">
                <tr valign="top">
                <th scope="row">OpenAI API Key</th>
                <td><input type="text" name="openai_api_key" value="<?php echo esc_attr(get_option('openai_api_key')); ?>" /></td>
                </tr>
                 
                <tr valign="top">
                <th scope="row">OpenAI Organization ID</th>
                <td><input type="text" name="openai_organization_id" value="<?php echo esc_attr(get_option('openai_organization_id')); ?>" /></td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
        </div>
        <?php
    }

    public function register_settings() {
        register_setting('openai_yoast_seo', 'openai_api_key');
        register_setting('openai_yoast_seo', 'openai_organization_id');
    }
}

// Instantiate the plugin class.
new OpenAI_Yoast_SEO();
