<?php
/**
 * Plugin Name: Simple OpenAI Internal Link Generator
 * Description: A simplified plugin to generate internal links using OpenAI for WordPress content.
 * Version: 1.0
 * Author: Capital District Digital
 */

if (!defined('ABSPATH')) {
    exit;
}

class Simple_OpenAI_Link_Generator {
    private $openai_api_key;
    private $openai_organization_id;

    public function __construct() {
        $this->openai_api_key = get_option('simple_openai_api_key');
        $this->openai_organization_id = get_option('simple_openai_organization_id');

        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_generate_links', array($this, 'handle_ajax_request'));
        add_action('edit_form_after_title', array($this, 'add_generate_button'));
        add_action('admin_menu', array($this, 'add_settings_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }

    public function enqueue_scripts() {
        wp_enqueue_script(
            'simple-openai-link-script', 
            plugins_url('js/simple-script.js', __FILE__), 
            array('jquery'), 
            '1.0.0', 
            true 
        );

        wp_localize_script(
            'simple-openai-link-script',
            'simple_openai_params',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('simple_openai_nonce')
            )
        );
    }

    public function add_generate_button() {
        echo '<button id="simple-generate-links" class="button">Generate Links</button>';
    }

    public function handle_ajax_request() {
        check_ajax_referer('simple_openai_nonce', 'nonce');

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        if ($post_id <= 0) {
            wp_send_json_error('Invalid Post ID');
            return;
        }

        $post_content = get_post_field('post_content', $post_id);
        if (!$post_content) {
            wp_send_json_error('No content found');
            return;
        }

        $suggestions = $this->get_link_suggestions($post_content);
        if (isset($suggestions['error'])) {
            wp_send_json_error($suggestions['error']);
        } else {
            wp_send_json_success($suggestions);
        }
    }

    private function get_link_suggestions($content) {
        $api_url = 'https://api.openai.com/v1/completions';

        $headers = array(
            'Authorization' => 'Bearer ' . $this->openai_api_key,
            'Content-Type'  => 'application/json',
            'OpenAI-Organization' => $this->openai_organization_id
        );

        $body = wp_json_encode(array(
            'model' => 'text-davinci-003',
            'prompt' => "Suggest internal links for: {$content}",
            'max_tokens' => 150
        ));

        $response = wp_remote_post($api_url, array(
            'headers' => $headers,
            'body'    => $body,
            'timeout' => 60
        ));

        if (is_wp_error($response)) {
            return array('error' => $response->get_error_message());
        }

        $response_body = json_decode(wp_remote_retrieve_body($response), true);
        return $response_body['choices'][0]['text'] ?? array('error' => 'No response from OpenAI.');
    }

    public function add_settings_menu() {
        add_menu_page(
            'Simple OpenAI Settings',
            'Simple OpenAI',
            'manage_options',
            'simple_openai_settings',
            array($this, 'settings_page_html'),
            null,
            99
        );
    }

public function settings_page_html() {
    if (!current_user_can('manage_options')) {
        return;
    }

    ?>
    <div class="wrap">
        <h1>Simple OpenAI Settings</h1>
        <form action="options.php" method="post">
            <?php
            settings_fields('simple_openai');
            do_settings_sections('simple_openai_settings');
            submit_button('Save Settings');
            ?>
        </form>
    </div>
    <?php
}

    public function register_settings() {
    register_setting('simple_openai', 'simple_openai_api_key');
    register_setting('simple_openai', 'simple_openai_organization_id');

    add_settings_section(
        'simple_openai_main_section',
        'OpenAI API Settings',
        null,
        'simple_openai_settings'
    );

    add_settings_field(
        'simple_openai_api_key_field',
        'OpenAI API Key',
        array($this, 'render_api_key_field'),
        'simple_openai_settings',
        'simple_openai_main_section'
    );

    add_settings_field(
        'simple_openai_organization_id_field',
        'OpenAI Organization ID',
        array($this, 'render_organization_id_field'),
        'simple_openai_settings',
        'simple_openai_main_section'
    );
}

public function render_api_key_field() {
    $api_key = get_option('simple_openai_api_key');
    echo '<input type="text" id="simple_openai_api_key" name="simple_openai_api_key" value="' . esc_attr($api_key) . '"/>';
}

public function render_organization_id_field() {
    $organization_id = get_option('simple_openai_organization_id');
    echo '<input type="text" id="simple_openai_organization_id" name="simple_openai_organization_id" value="' . esc_attr($organization_id) . '"/>';
}

}

new Simple_OpenAI_Link_Generator();
